unit XBBDemo1;

{
  Demonstration of XML Building Blocks.

  Copyright  Keith Wood (kbwood@iprimus.com.au)
  7 March 2002.
}

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, ComCtrls, Buttons, ExtCtrls, Grids, DB, DBTables, Menus,
  OleCtrls, SHDocVw, XMLDOM, XMLBlocks, XBBDemo2;

type
  TfrmXBB = class(TForm)
    pgcViews: TPageControl;
      tabTree: TTabSheet;
        trvXML: TTreeView;
        splTree: TSplitter;
        stgXML: TStringGrid;
      tabMemo: TTabSheet;
        memXML: TMemo;
      tabWriter: TTabSheet;
        memWriter: TMemo;
      tabWebBrowser: TTabSheet;
        webBrowser: TWebBrowser;
      tabComponent: TTabSheet;
    Panel1: TPanel;
      Shape1: TShape;
      Shape2: TShape;
      Shape3: TShape;
      Shape4: TShape;
      Shape5: TShape;
      btnParser: TButton;
      chkParser: TCheckBox;
      btnSQL: TButton;
      chkSQL: TCheckBox;
      btnTextFile: TButton;
      chkTextFile: TCheckBox;
      btnComponent: TButton;
      chkComponent: TCheckBox;
      btnMerge: TBitBtn;
      btnTimestamp: TBitBtn;
      chkTimestamp: TCheckBox;
      btnTransform: TBitBtn;
      chkTransform: TCheckBox;
      stxFork: TStaticText;
      stxTreeView: TStaticText;
      chkTreeView: TCheckBox;
      stxMemo: TStaticText;
      chkMemo: TCheckBox;
      btnWriter: TButton;
      chkWriter: TCheckBox;
      chkWebBrowser: TCheckBox;
      btnGo: TBitBtn;
    xbbParser: TXBBParser;
    xbbSQL: TXBBSQL;
    xbbTextFile: TXBBTextFile;
    xbbComponent: TXBBComponent;
    xbbMerge: TXBBMerge;
    xbbTimestamp: TXBBTimestamp;
    xbbTransform: TXBBTransform;
    xbbFork: TXBBFork;
    xbbTreeView: TXBBTreeView;
    xbbStringGrid: TXBBStringGrid;
    xbbMemo: TXBBMemo;
    xbbWriter: TXBBWriter;
    xbbWebBrowser: TXBBWebBrowser;
    xbbComponentCreate: TXBBComponentCreate;
    dlgOpen: TOpenDialog;
    dlgSave: TSaveDialog;
    popStringGrid: TPopupMenu;
      mniSingleNode: TMenuItem;
    popTree: TPopupMenu;
      mniExpand: TMenuItem;
      mniCollapse: TMenuItem;
    stxWebBrowser: TStaticText;
    procedure FormCreate(Sender: TObject);
    procedure btnParserClick(Sender: TObject);
    procedure btnSQLClick(Sender: TObject);
    procedure xbbSQLRecordTagCreate(Sender: TObject; Element: IDOMElement;
      Dataset: TDataSet);
    procedure btnTextFileClick(Sender: TObject);
    procedure btnComponentClick(Sender: TObject);
    procedure btnMergeClick(Sender: TObject);
    procedure xbbMergeTagCreate(Sender: TObject; Element: IDOMElement);
    procedure btnTimestampClick(Sender: TObject);
    procedure btnTransformClick(Sender: TObject);
    procedure btnWriterClick(Sender: TObject);
    procedure mniSingleNodeClick(Sender: TObject);
    procedure btnGoClick(Sender: TObject);
    procedure mniExpandCollapseClick(Sender: TObject);
    procedure trvXMLChange(Sender: TObject; TreeNode: TTreeNode);
  private
    FLastTreeNode: TTreeNode;
    FPropertiesDialog: TfrmProperties;
  public
  end;

var
  frmXBB: TfrmXBB;

implementation

{$R *.dfm}

{ Initialisation }
procedure TfrmXBB.FormCreate(Sender: TObject);
begin
  trvXML.Images         := DefaultNodeImages;
  btnParser.Hint        := xbbParser.XMLSource;
  btnSQL.Hint           := xbbSQL.SQL.Text;
  btnTextFile.Hint      := xbbTextFile.FileName;
  btnComponent.Hint     := xbbComponent.Component.Name;
  btnMerge.Hint         := xbbMerge.TagName;
  btnTimestamp.Hint     := xbbTimestamp.Format;
  btnTransform.Hint     := xbbTransform.XSLSource;
  btnWriter.Hint        := xbbWriter.FileName;
  mniSingleNode.Checked := xbbStringGrid.SingleNode;
  FPropertiesDialog     := TfrmProperties.Create(Self);
end;

{ Ask which XML document to open }
procedure TfrmXBB.btnParserClick(Sender: TObject);
begin
  with dlgOpen do
  begin
    DefaultExt := 'xml';
    FileName   := xbbParser.XMLSource;
    Filter     := 'XML files (*.xml)|*.xml|All files (*.*)|*.*';
    Options    := [ofHideReadOnly, ofPathMustExist, ofFileMustExist,
                  ofEnableSizing];
    Title      := 'Open XML Document';
    if Execute then
      xbbParser.XMLSource := FileName;
  end;
  btnParser.Hint := xbbParser.XMLSource;
end;

{ Ask for a SQL query to execute }
procedure TfrmXBB.btnSQLClick(Sender: TObject);
var
  TagName, RecordTagName, DatabaseName, SQL: string;
  Format: TXBBFieldFormat;
begin
  TagName       := xbbSQL.TagName;
  RecordTagName := xbbSQL.RecordTagName;
  DatabaseName  := xbbSQL.DatabaseName;
  SQL           := xbbSQL.SQL.Text;
  Format        := xbbSQL.FieldFormat;
  FPropertiesDialog.
    SQLProperties(TagName, RecordTagName, DatabaseName, SQL, Format);
  xbbSQL.TagName       := TagName;
  xbbSQL.RecordTagName := RecordTagName;
  xbbSQL.DatabaseName  := DatabaseName;
  xbbSQL.SQL.Text      := SQL;
  xbbSQL.FieldFormat   := Format;
  btnSQL.Hint          := SQL;
end;

{ Set an attribute on each record element }
procedure TfrmXBB.xbbSQLRecordTagCreate(Sender: TObject;
  Element: IDOMElement; Dataset: TDataSet);
begin
  Element.SetAttribute('id', Dataset.Fields[0].DisplayText);
end;

{ Ask which text file to include }
procedure TfrmXBB.btnTextFileClick(Sender: TObject);
begin
  with dlgOpen do
  begin
    DefaultExt := 'txt';
    FileName   := xbbTextFile.FileName;
    Filter     := 'Text files (*.txt)|*.txt|All files (*.*)|*.*';
    Options    := [ofHideReadOnly, ofPathMustExist, ofFileMustExist,
                  ofEnableSizing];
    Title      := 'Open Text File';
    if Execute then
      xbbTextFile.FileName := FileName;
  end;
  btnTextFile.Hint := xbbTextFile.FileName;
end;

{ Ask which component to dump }
procedure TfrmXBB.btnComponentClick(Sender: TObject);
var
  TagName, CompName: string;
begin
  TagName  := xbbComponent.TagName;
  CompName := xbbComponent.Component.Name;
  FPropertiesDialog.ComponentProperties(TagName, CompName);
  xbbComponent.TagName := TagName;
  if CompName = Self.Name then
    xbbComponent.Component := Self
  else
    xbbComponent.Component := FindComponent(CompName);
  btnComponent.Hint := xbbComponent.Component.Name;
end;

{ Ask for the main tag name for the merged document }
procedure TfrmXBB.btnMergeClick(Sender: TObject);
var
  TagName: string;
begin
  TagName := xbbMerge.TagName;
  FPropertiesDialog.SimpleProperties(TagName);
  xbbMerge.TagName := TagName;
  btnMerge.Hint    := xbbMerge.TagName;
end;

{ Set an attribute on the merged document element }
procedure TfrmXBB.xbbMergeTagCreate(Sender: TObject; Element: IDOMElement);
begin
  Element.SetAttribute('docs', IntToStr(xbbMerge.NumDocuments));
end;

{ Ask for the format of the timestamp }
procedure TfrmXBB.btnTimestampClick(Sender: TObject);
var
  TagName, Format: string;
  InsertAtStart: Boolean;
begin
  TagName       := xbbTimestamp.TagName;
  Format        := xbbTimestamp.Format;
  InsertAtStart := xbbTimestamp.InsertAtStart;
  FPropertiesDialog.TimestampProperties(TagName, Format, InsertAtStart);
  xbbTimestamp.TagName       := TagName;
  xbbTimestamp.Format        := Format;
  xbbTimestamp.InsertAtStart := InsertAtStart;
  btnTimestamp.Hint          := xbbTimestamp.Format;
end;

{ Ask which XSL transformation to use }
procedure TfrmXBB.btnTransformClick(Sender: TObject);
begin
  with dlgOpen do
  begin
    DefaultExt := 'xsl';
    FileName   := xbbTransform.XSLSource;
    Filter     := 'XSL files (*.xsl)|*.xsl|All files (*.*)|*.*';
    Options    := [ofHideReadOnly, ofPathMustExist, ofFileMustExist,
                  ofEnableSizing];
    Title      := 'Open Transformation';
    if Execute then
      xbbTransform.XSLSource := FileName;
  end;
  btnTransform.Hint := xbbTransform.XSLSource;
end;

{ Ask where to save the output }
procedure TfrmXBB.btnWriterClick(Sender: TObject);
begin
  with dlgSave do
  begin
    DefaultExt := 'xml';
    FileName   := xbbWriter.FileName;
    Filter     := 'XML files (*.xml)|*.xml|All files (*.*)|*.*';
    Options    := [ofOverwritePrompt, ofHideReadOnly, ofPathMustExist,
                  ofEnableSizing];
    Title      := 'Save XML Document';
    if Execute then
      xbbWriter.FileName := FileName;
  end;
  btnWriter.Hint := xbbWriter.FileName;
end;

{ Toggle the output style of the string grid building block }
procedure TfrmXBB.mniSingleNodeClick(Sender: TObject);
begin
  mniSingleNode.Checked    := not mniSingleNode.Checked;
  xbbStringGrid.SingleNode := mniSingleNode.Checked;
  FLastTreeNode            := nil;
end;

{ Start the whole process after linking building blocks together }
procedure TfrmXBB.btnGoClick(Sender: TObject);
var
 xbbConsumer: IXMLConsumer;
begin
  trvXML.Items.Clear;
  stgXML.ColCount := 2;
  stgXML.RowCount := 2;
  stgXML.Rows[1].Clear;
  memXML.Lines.Clear;
  memWriter.Lines.Clear;

  { Validate inputs and outputs }
  if not (chkParser.Checked or chkSQL.Checked or
    chkTextFile.Checked or chkComponent.Checked) then
  begin
    MessageDlg('Please select one or more XML sources', mtError, [mbOK], 0);
    Exit;
  end;
  if not (chkTreeView.Checked or chkMemo.Checked or
    chkWriter.Checked or chkWebBrowser.Checked) then
  begin
    MessageDlg('Please select one or more XML destinations', mtError, [mbOK], 0);
    Exit;
  end;

  xbbParser.Consumer    := xbbMerge;
  xbbSQL.Consumer       := xbbMerge;
  xbbTextFile.Consumer  := xbbMerge;
  xbbComponent.Consumer := xbbMerge;
  { Prepare merge building block - if necessary }
  xbbMerge.NumDocuments := Ord(chkParser.Checked) +
    Ord(chkSQL.Checked) + Ord(chkTextFile.Checked) + Ord(chkComponent.Checked);
  if chkTimestamp.Checked then
  begin
    xbbConsumer := xbbTimestamp;
    if chkTransform.Checked then
      xbbTimestamp.Consumer := xbbTransform
    else
      xbbTimestamp.Consumer := xbbFork;
  end
  else if chkTransform.Checked then
    xbbConsumer := xbbTransform
  else
    xbbConsumer := xbbFork;
  if xbbMerge.NumDocuments = 1 then
  begin
    if chkParser.Checked then
      xbbParser.Consumer := xbbConsumer
    else if chkSQL.Checked then
      xbbSQL.Consumer := xbbConsumer
    else if chkTextFile.Checked then
      xbbTextFile.Consumer := xbbConsumer
    else
      xbbComponent.Consumer := xbbConsumer;
  end
  else
    xbbMerge.Consumer := xbbConsumer;

  { Prepare fork building block }
  xbbFork.Consumers.Clear;
  if chkTreeView.Checked then
    xbbFork.Consumers.Add.Consumer := xbbTreeView;
  if chkMemo.Checked then
    xbbFork.Consumers.Add.Consumer := xbbMemo;
  if chkWriter.Checked then
    xbbFork.Consumers.Add.Consumer := xbbWriter;
  if chkWebBrowser.Checked then
    xbbFork.Consumers.Add.Consumer := xbbWebBrowser;
  if chkComponent.Checked then
    xbbFork.Consumers.Add.Consumer := xbbComponentCreate;

  { And start the whole process }
  if chkParser.Checked then
    xbbParser.CreateDocument;
  if chkSQL.Checked then
    xbbSQL.CreateDocument;
  if chkTextFile.Checked then
    xbbTextFile.CreateDocument;
  if chkComponent.Checked then
    xbbComponent.CreateDocument;

  { Display results }
  pgcViews.ActivePage := tabTree;
  if not chkTreeView.Checked then
  begin
    if chkMemo.Checked then
      pgcViews.ActivePage := tabMemo
    else if chkWriter.Checked then
      pgcViews.ActivePage := tabWriter
    else if chkWebBrowser.Checked then
      pgcViews.ActivePage := tabWebBrowser
    else if chkComponent.Checked then
      pgcViews.ActivePage := tabComponent;
  end;
  if chkWriter.Checked then
    memWriter.Lines.LoadFromFile(xbbWriter.FileName);
  if chkComponent.Checked and (xbbComponentCreate.Component is TControl) then
    TControl(xbbComponentCreate.Component).Parent := tabComponent;
end;

{ Expand/collapse a node in the tree and all its children }
procedure TfrmXBB.mniExpandCollapseClick(Sender: TObject);
begin
  with trvXML do
    if Assigned(Selected) then
    begin
      Items.BeginUpdate;
      if TMenuItem(Sender).Tag = 1 then
        Selected.Expand(True)
      else
        Selected.Collapse(True);
      TopItem := Selected;
      Items.EndUpdate;
    end;
end;

{ Display a node in the string grid }
procedure TfrmXBB.trvXMLChange(Sender: TObject; TreeNode: TTreeNode);
var
  Document: IDOMDocument;
  Node: IDOMNode;
begin
  if Assigned(TreeNode) then
  begin
    if FLastTreeNode <> TreeNode then
    begin
      Node := TXBBNodePointer(TreeNode.Data).Node;
      case Node.NodeType of
        DOCUMENT_NODE:
          Document := Node as IDOMDocument;
        ELEMENT_NODE:
          begin
            Document := GetDOM().CreateDocument('', '', nil);
            Document.AppendChild(Node.CloneNode(True));
          end;
        else
          Exit;
      end;
      FLastTreeNode := TreeNode;
      xbbStringGrid.DocumentReady(Document);
    end;
  end;
end;

initialization
  { Register class for retrieval from streams }
  RegisterClass(TBitBtn);
  RegisterClass(TButton);
  RegisterClass(TCheckBox);
  RegisterClass(TLabel);
  RegisterClass(TMemo);
  RegisterClass(TMenuItem);
  RegisterClass(TOpenDialog);
  RegisterClass(TPageControl);
  RegisterClass(TPanel);
  RegisterClass(TPopupMenu);
  RegisterClass(TSaveDialog);
  RegisterClass(TShape);
  RegisterClass(TSplitter);
  RegisterClass(TStaticText);
  RegisterClass(TStringGrid);
  RegisterClass(TTabSheet);
  RegisterClass(TTreeView);
  RegisterClass(TWebBrowser);
  RegisterClass(TXBBComponent);
  RegisterClass(TXBBComponentCreate);
  RegisterClass(TXBBFork);
  RegisterClass(TXBBMemo);
  RegisterClass(TXBBMerge);
  RegisterClass(TXBBParser);
  RegisterClass(TXBBSQL);
  RegisterClass(TXBBStringGrid);
  RegisterClass(TXBBTextFile);
  RegisterClass(TXBBTimestamp);
  RegisterClass(TXBBTransform);
  RegisterClass(TXBBTreeView);
  RegisterClass(TXBBWebBrowser);
  RegisterClass(TXBBWriter);
end.
